package gov.va.med.mhv.common.api.transfer;

import gov.va.med.mhv.common.api.util.ClientApplicationHelper;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.util.Date;
import java.util.StringTokenizer;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.time.DateFormatUtils;

public class Session implements Serializable {
	private static final long serialVersionUID = -5454873976482094676L;
	public static String RFC1123_PATTERN = "EEE, dd MMM yyyy HH:mm:ss z";

	private long userId;
	private String delegateeUserName;
	private String delegateeName;
	private String delegateeAccess;
	private Date timestamp;
	private ClientApplication clientApplication;
	private int expirationSeconds = 300; // Default to 5 minutes

	public Session(long userId, ClientApplication clientApplication) {
		this.userId = userId;
		this.clientApplication = clientApplication;
		this.expirationSeconds = clientApplication.getTimeoutSeconds().intValue();
		touch();
	}
	
	private Session(long userId, ClientApplication clientApplication, Date timestamp) {
		this(userId, clientApplication);
		this.expirationSeconds = clientApplication.getTimeoutSeconds().intValue();
		this.timestamp = timestamp;
	}
	
	public Session(long userId, String delegateeUserName, String delegateeName, ClientApplication clientApplication) {
		this(userId, clientApplication);
		this.delegateeUserName = delegateeUserName;
		this.delegateeName = delegateeName;
	}
	
	public Session(long userId, String delegateeUserName, String delegateeName, String delegateeAccess, ClientApplication clientApplication) {
		this(userId, clientApplication);
		this.delegateeUserName = delegateeUserName;
		this.delegateeName = delegateeName;
		this.delegateeAccess = delegateeAccess;
	}
	
	private Session(long userId, String delegateeUserName, String delegateeName, ClientApplication clientApplication, Date timestamp) {
		this(userId, clientApplication, timestamp);
		this.delegateeUserName = delegateeUserName;
		this.delegateeName = delegateeName;
	}
	
	private Session(long userId, String delegateeUserName, String delegateeName, String delegateeAccess, ClientApplication clientApplication, Date timestamp) {
		this(userId, clientApplication, timestamp);
		this.delegateeUserName = delegateeUserName;
		this.delegateeName = delegateeName;
		this.delegateeAccess = delegateeAccess;
	}

	public long getUserId() {
		return userId;
	}

	public void setUserId(long userId) {
		this.userId = userId;
	}
	
	public String getDelegateeUserName() {
		return delegateeUserName;
	}
	
	public String getDelegateeName() {
		return delegateeName;
	}

	
	public String getDelegateeAccess() {
		return delegateeAccess;
	}
	
	public Date getTimestamp() {
		return timestamp;
	}

	public void setTimestamp(Date timestamp) {
		this.timestamp = timestamp;
	}

	public long getExpirationSeconds() {
		return expirationSeconds;
	}

	public void setExpirationSeconds(int expirationSeconds) {
		this.expirationSeconds = expirationSeconds;
	}

	public ClientApplication getClientApplication() {
		return clientApplication;
	}

	public void setClientApplication(ClientApplication clientApplication) {
		this.clientApplication = clientApplication;
	}

	public String getFormattedTimestamp() {
		return DateFormatUtils.formatUTC(timestamp, RFC1123_PATTERN);
	}

	public void touch() {
		this.timestamp = new Date(System.currentTimeMillis() + (getExpirationSeconds() * 1000));
	}

	public boolean isExpired() {
		long currentTimeinSec = TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis());
		long timeSeconds = TimeUnit.MILLISECONDS.toSeconds(timestamp.getTime());
		if ((currentTimeinSec - timeSeconds) <= (getExpirationSeconds())) {
			return false;
		} else {
			return true;
		}
	}

	/**
	 * 
	 * @return String of userId|patientId|clientApplication|Expiration
	 * @throws IOException
	 */
	public byte[] marshall() throws IOException {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		baos.write(String.valueOf(getUserId()).getBytes());
		baos.write("|".getBytes());
		baos.write(String.valueOf(getClientApplication().getId()).getBytes());
		baos.write("|".getBytes());
		baos.write(String.valueOf(getTimestamp().getTime()).getBytes());
		
		if(this.delegateeUserName != null) {
			baos.write("|".getBytes());
			baos.write(getDelegateeUserName().getBytes());
			baos.write("|".getBytes());
			baos.write(String.valueOf(getDelegateeName()).getBytes());
		}
		if(this.delegateeAccess != null) {
			baos.write("|".getBytes());
			baos.write(String.valueOf(getDelegateeAccess()).getBytes());
		}
		
		return baos.toByteArray();
	}


	/**
	 * 
	 * Parses String of userId|clientApplication|Expiration
	 * @return Session object
	 * @throws IOException
	 */
	public static Session unmarshall(byte[] bytes) throws IOException,
			ClassNotFoundException {
		String sessionStr = new String(bytes);
		StringTokenizer st = new StringTokenizer(sessionStr, "|");
		if (st.countTokens() == 3) {
			try {
				return new Session(Long.parseLong(st.nextToken()),
								   ClientApplicationHelper.findClientApplication(Long.parseLong(st.nextToken())), 
								   new Date(Long.parseLong(st.nextToken())));
			} catch (Exception e) {
				throw new IOException("Unable to unmarshall object");
			}
		} else if (st.countTokens() == 6) { // delegatee Access information included...

			Long userId = Long.parseLong(st.nextToken());
			ClientApplication clientApplication = ClientApplicationHelper.findClientApplication(Long.parseLong(st.nextToken()));
			Date timestamp = new Date(Long.parseLong(st.nextToken()));
			String delegateeUserName = st.nextToken();
			String delegateeName = st.nextToken();
			String delegateeAccess = st.nextToken();
			
			return new Session(userId,
					   delegateeUserName, 
					   delegateeName,
					   delegateeAccess,
					   clientApplication,
					   timestamp);
		}
		else {
			throw new IOException("Unable to unmarshall object");
		}
	}

	public String toString() {
		return "Session {" + "userId: " + userId + ";"  + "expirationTimestamp: " + timestamp + ";"
				+ "clientApplication: " + clientApplication + ";" 
				+ "delegateeId: " + delegateeUserName + ";" + "delegateeName: " + delegateeName + ";" 
				+ "delegateeAccess: " + delegateeAccess + ";" + "}";
	}
	
	

}
